#!/usr/bin/env python3
"""
Simple example of using Anthropic API through a custom proxy endpoint.
"""
import yaml
from anthropic import Anthropic

def load_config(config_path="config.yaml"):
    """Load configuration from YAML file."""
    try:
        with open(config_path, 'r') as f:
            config = yaml.safe_load(f)
        return config
    except FileNotFoundError:
        raise FileNotFoundError(
            f"Config file '{config_path}' not found. "
            "Please create it based on config.yaml.example"
        )

def main():
    # Load configuration
    config = load_config()

    # Validate required config fields
    if not config.get('api_key'):
        raise ValueError("api_key is required in config.yaml")
    if not config.get('base_url'):
        raise ValueError("base_url is required in config.yaml")

    # Initialize Anthropic client with custom base URL
    client = Anthropic(
        api_key=config['api_key'],
        base_url=config['base_url']
    )

    # Create a message
    message = client.messages.create(
        model="claude-sonnet-4-5-20250929",
        max_tokens=1024,
        messages=[
            {
                "role": "user",
                "content": "Hello! Can you tell me a short interesting fact about artificial intelligence?"
            }
        ]
    )

    # Print the response
    print("Response from Claude:")
    print("-" * 50)
    print(message.content[0].text)
    print("-" * 50)
    print(f"\nModel used: {message.model}")
    print(f"Tokens used: {message.usage.input_tokens} input, {message.usage.output_tokens} output")

if __name__ == "__main__":
    main()
